// Full working version of index.js with logging, fixed lastBuyPrice tracking, and improved autoTrade feedback

const ethers = require('ethers');
const fs = require('fs');
const { Telegraf } = require('telegraf');
require('dotenv').config();

const bot = new Telegraf(process.env.BOT_TOKEN);
let botChatId = null;

const provider = new ethers.providers.JsonRpcProvider(process.env.INFURA_URL);
const wallet = new ethers.Wallet(process.env.PRIVATE_KEY, provider);

const routerAddress = process.env.UNISWAP_ROUTER || '0x7a250d5630B4cF539739dF2C5dAcb4c659F2488D';
const WETH = process.env.WETH_ADDRESS || '0xC02aaA39b223FE8D0A0e5C4F27eAD9083C756Cc2';
const tokenAddress = process.env.TOKEN_ADDRESS;

const routerABI = require('./abis/uniswap_router.json');
const tokenABI = [
    "function decimals() view returns (uint8)",
    "function symbol() view returns (string)",
    "function balanceOf(address) view returns (uint256)",
    "function allowance(address owner, address spender) view returns (uint256)",
    "function approve(address spender, uint256 amount) returns (bool)"
];

const router = new ethers.Contract(routerAddress, routerABI, wallet);

let sellMultiplier = 1.5;
let interval;

let lastBuyPrice = 0;
const lastBuyPriceFile = './lastBuyPrice.json';

try {
    const data = JSON.parse(fs.readFileSync(lastBuyPriceFile));
    lastBuyPrice = parseFloat(data.price || 0);
} catch (e) {
    console.warn('⚠️ Could not read lastBuyPrice.json, defaulting to 0');
}

function saveLastBuyPrice(price) {
    fs.writeFileSync(lastBuyPriceFile, JSON.stringify({ price }, null, 2));
    lastBuyPrice = price;
    console.log(`💾 Updated lastBuyPrice.json to ${price}`);
}

async function getTokenBalance() {
    const token = new ethers.Contract(tokenAddress, tokenABI, provider);
    return await token.balanceOf(wallet.address);
}

async function getETHBalance() {
    return await provider.getBalance(wallet.address);
}

async function autoTrade() {
    console.log('⏳ Running autoTrade()...');
    try {
        const token = new ethers.Contract(tokenAddress, tokenABI, wallet);
        const balance = await token.balanceOf(wallet.address);

        if (balance.isZero()) {
            console.log('ℹ️ No token balance. Skipping.');
            return;
        }

        const ethOut = await router.getAmountsOut(balance, [tokenAddress, WETH]);
        const currentPrice = parseFloat(ethers.utils.formatEther(ethOut[1]));
        const targetSellPrice = lastBuyPrice * sellMultiplier;

        console.log(`📈 Current Price: ${currentPrice}, Target: ${targetSellPrice}`);

        if (currentPrice >= targetSellPrice) {
            const ethBalance = await getETHBalance();
            const gasReserve = ethers.utils.parseUnits("0.0025", "ether");

            if (ethBalance.lt(gasReserve)) {
                if (botChatId) await bot.telegram.sendMessage(botChatId, "❌ Not enough ETH to cover gas. Auto-sell paused.");
                console.log('❌ Not enough ETH for gas.');
                return;
            }

            const allowance = await token.allowance(wallet.address, routerAddress);
            if (allowance.lt(balance)) {
                const approveTx = await token.approve(routerAddress, balance);
                await approveTx.wait();
                console.log('✅ Approved tokens for sell.');
            }

            const deadline = Math.floor(Date.now() / 1000) + 600;
            const slippage = 0.05;
            const minOut = ethOut[1].sub(ethOut[1].mul(Math.floor(slippage * 100)).div(100));

            const tx = await router.swapExactTokensForETHSupportingFeeOnTransferTokens(
                balance, minOut, [tokenAddress, WETH], wallet.address, deadline
            );
            await tx.wait();

            const profit = ((currentPrice / lastBuyPrice - 1) * 100).toFixed(2);
            if (botChatId) await bot.telegram.sendMessage(botChatId, `✅ Sold for profit (+${profit}%)\nTx: ${tx.hash}`);
            console.log(`✅ Auto-sold. Profit: +${profit}%, Tx: ${tx.hash}`);
        } else {
            console.log('🔍 Price below target. Waiting...');
        }
    } catch (err) {
        console.error('❌ Auto-trade error:', err);
        if (botChatId) await bot.telegram.sendMessage(botChatId, `❌ Auto-trade error:\n${err.reason || err.message || err}`);
    }
}

bot.command('start', (ctx) => {
    botChatId = ctx.chat.id;
    ctx.reply('🤖 Auto-trading started.');
    if (!interval) interval = setInterval(autoTrade, 30000);
    console.log('✅ Bot started. autoTrade() running every 30s');
});

bot.command('setsell', (ctx) => {
    const parts = ctx.message.text.split(' ');
    if (parts.length === 2) {
        const val = parseFloat(parts[1]);
        if (!isNaN(val) && val > 0) {
            sellMultiplier = 1 + val;
            ctx.reply(`✅ Sell multiplier set to ${sellMultiplier.toFixed(2)} (i.e., ${val * 100}% profit)`);
        } else {
            ctx.reply('❌ Enter a number > 0 (e.g., 0.3 for 30%)');
        }
    } else {
        ctx.reply('Usage: /setsell 0.3');
    }
});

bot.command('balance', async(ctx) => {
    try {
        const ethBalance = await getETHBalance();
        const ethFormatted = ethers.utils.formatEther(ethBalance);
        const token = new ethers.Contract(tokenAddress, tokenABI, provider);
        const decimals = await token.decimals();
        const symbol = await token.symbol();
        const rawTokenBalance = await token.balanceOf(wallet.address);
        const tokenBalance = ethers.utils.formatUnits(rawTokenBalance, decimals);

        ctx.reply(`Your balances:\nETH: ${parseFloat(ethFormatted).toFixed(6)}\n${symbol}: ${parseFloat(tokenBalance).toFixed(6)} ${symbol}`);
    } catch (e) {
        ctx.reply(`Balance fetch error: ${e.message || e}`);
    }
});

bot.command('buyusd', async(ctx) => {
    try {
        const parts = ctx.message.text.split(' ');
        const usdAmount = parseFloat(parts[1]);
        if (isNaN(usdAmount) || usdAmount <= 0) return ctx.reply('Use: /buyusd 1');

        const ethPriceUSD = 2000;
        const ethAmount = usdAmount / ethPriceUSD;
        const ethBalance = await wallet.getBalance();
        const gasReserve = ethers.utils.parseEther("0.002");

        if (ethBalance.lte(gasReserve)) {
            return ctx.reply(`❌ Not enough ETH to buy after reserving gas. You have ${ethers.utils.formatEther(ethBalance)} ETH`);
        }

        const spendable = ethBalance.sub(gasReserve);
        const actualBuyAmount = ethers.utils.parseEther(Math.min(ethAmount, parseFloat(ethers.utils.formatEther(spendable))).toString());

        const deadline = Math.floor(Date.now() / 1000) + 600;
        const tx = await router.swapExactETHForTokensSupportingFeeOnTransferTokens(
            0, [WETH, tokenAddress], wallet.address, deadline, { value: actualBuyAmount }
        );

        await tx.wait();

        const token = new ethers.Contract(tokenAddress, tokenABI, provider);
        const balance = await token.balanceOf(wallet.address);
        const ethOut = await router.getAmountsOut(balance, [tokenAddress, WETH]);
        const price = parseFloat(ethers.utils.formatEther(ethOut[1])) / parseFloat(ethers.utils.formatUnits(balance, await token.decimals()));

        saveLastBuyPrice(price);

        ctx.reply(`✅ Bought tokens with ~${ethers.utils.formatEther(actualBuyAmount)} ETH\nTx: ${tx.hash}`);
    } catch (error) {
        ctx.reply(`Buy failed: ${error.reason || error.message}`);
    }
});

bot.launch().then(() => console.log('🚀 Telegram bot launched.'));